package de.davidartmann.rosa.fragment;

import android.content.Context;
import android.content.DialogInterface;
import android.graphics.drawable.ColorDrawable;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.support.v4.content.ContextCompat;
import android.support.v4.view.MenuItemCompat;
import android.support.v7.app.ActionBar;
import android.support.v7.app.AlertDialog;
import android.support.v7.widget.RecyclerView;
import android.support.v7.widget.SearchView;
import android.support.v7.widget.StaggeredGridLayoutManager;
import android.support.v7.widget.helper.ItemTouchHelper;
import android.support.v7.widget.helper.ItemTouchHelper.Callback;
import android.view.LayoutInflater;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.view.inputmethod.InputMethodManager;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import de.davidartmann.rosa.R;
import de.davidartmann.rosa.activity.MainListActivity;
import de.davidartmann.rosa.adapter.MainListAdapter;
import de.davidartmann.rosa.db.async.DeletePersonsInDb;
import de.davidartmann.rosa.db.async.FindAllActivePersonsFromDb;
import de.davidartmann.rosa.db.async.SetPositionByNumberOfSoftDeletedPersonInDb;
import de.davidartmann.rosa.db.async.SoftDeletePersonsInDb;
import de.davidartmann.rosa.model.Person;

/**
 * Fragment for the main listview of all persons.
 * Created by david on 03.03.16.
 */
public class MainListFragment extends Fragment
        implements MainListAdapter.IMainFragmentLongClickMenu,
        SoftDeletePersonsInDb.ISoftDeletePersons, DeletePersonsInDb.IDeletePersons,
        FindAllActivePersonsFromDb.IFindAllActive,
        SetPositionByNumberOfSoftDeletedPersonInDb.ISetPosBySoftDeleted {

    private MainListAdapter.IMainListAdapter iMainListAdapter;
    private boolean mIsLongClickActive;
    private List<Person> mPersonsToArchiveOrDelete;
    private IMainListFragment iMainListFragment;
    private int mNumberOfSoftDeleted;
    private MainListAdapter mMainListAdapter; //member for animations, as outstanding features

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);
        iMainListAdapter = (MainListAdapter.IMainListAdapter) context;
        iMainListFragment = (IMainListFragment) context;
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fragment_main_list, container, false);
        setHasOptionsMenu(true);
        RecyclerView recyclerView =
                (RecyclerView) view.findViewById(R.id.fragment_main_list_recyclerview);
        recyclerView.setHasFixedSize(true);
        recyclerView.setLayoutManager(
                new StaggeredGridLayoutManager(2, StaggeredGridLayoutManager.VERTICAL));
        mMainListAdapter = new MainListAdapter(R.layout.listview_viewholder_cardview,
                view.getContext(), iMainListAdapter, this);
        recyclerView.setAdapter(mMainListAdapter);
        InputMethodManager imm = (InputMethodManager)getActivity()
                .getSystemService(Context.INPUT_METHOD_SERVICE);
        imm.hideSoftInputFromWindow(view.getWindowToken(), 0);
        mIsLongClickActive = false;
        mPersonsToArchiveOrDelete = new ArrayList<>();
        return view;
    }

    @Override
    public void onCreateOptionsMenu(Menu menu, MenuInflater inflater) {
        super.onCreateOptionsMenu(menu, inflater);
        menu.clear();
        if (mIsLongClickActive) {
            if (mPersonsToArchiveOrDelete.size() == 1) {
                inflater.inflate(R.menu.menu_main_list_longclick_single, menu);
            } else {
                inflater.inflate(R.menu.menu_main_list_longclick, menu);
            }
            setActionBarBgColor(true);
        } else {
            inflater.inflate(R.menu.menu_main_list, menu);
            setActionBarBgColor(false);
            final MenuItem searchItem = menu.findItem(R.id.menu_main_search);
            final SearchView searchView = (SearchView) MenuItemCompat.getActionView(searchItem);
            searchView.setOnQueryTextListener(new SearchView.OnQueryTextListener() {
                @Override
                public boolean onQueryTextSubmit(String query) {
                    return false;
                }

                @Override
                public boolean onQueryTextChange(String newText) {
                    mMainListAdapter.search(newText);
                    return true;
                }
            });
        }
    }

    private void setActionBarBgColor(boolean b) {
        ActionBar actionBar = ((MainListActivity) getActivity()).getSupportActionBar();
        if (actionBar != null) {
            if (b) {
                actionBar.setBackgroundDrawable(new ColorDrawable(
                        ContextCompat.getColor(getContext(), R.color.colorDarkGray)));
            } else {
                actionBar.setBackgroundDrawable(new ColorDrawable(
                        ContextCompat.getColor(getContext(), R.color.colorPrimary)));
            }
        }
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        switch (item.getItemId()) {
            case R.id.menu_main_longclick_softdelete:
                showSoftDeleteChoosenPersonsDialog();
                return true;
            case R.id.menu_main_longclick_deleteforever:
                showDeleteChoosenPersonsForEverDialog();
                return true;
            case R.id.menu_main_longclick_single_edit:
                if (mPersonsToArchiveOrDelete.size() == 1) {
                    setActionBarBgColor(false);
                    iMainListFragment.onMenuEditButtonClicked(mPersonsToArchiveOrDelete.get(0));
                }
                return true;
        }
        return super.onOptionsItemSelected(item);
    }

    private void showDeleteChoosenPersonsForEverDialog() {
        final Context context = getContext();
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        if (mPersonsToArchiveOrDelete.size() == 1) {
            builder.setMessage(context.getString(R.string.dialog_delete_choosen_person_for_ever));
        } else {
            builder.setMessage(context.getString(R.string.dialog_delete_choosen_persons_for_ever));
        }
        builder.setPositiveButton(context.getString(R.string.Ja),
                new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                new DeletePersonsInDb(context, mPersonsToArchiveOrDelete, MainListFragment.this)
                        .execute();
                dialog.dismiss();
            }
        });
        builder.setNegativeButton(context.getString(R.string.Nein),
                new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        builder.show();
    }

    private void showSoftDeleteChoosenPersonsDialog() {
        final Context context = getContext();
        AlertDialog.Builder builder = new AlertDialog.Builder(context);
        if (mPersonsToArchiveOrDelete.size() == 1) {
            builder.setMessage(context.getString(R.string.dialog_soft_delete_choosen_person));
        } else {
            builder.setMessage(context.getString(R.string.dialog_soft_delete_choosen_persons));
        }
        builder.setPositiveButton(context.getString(R.string.Ja),
                new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                new SoftDeletePersonsInDb(context, MainListFragment.this, mPersonsToArchiveOrDelete)
                        .execute();
                dialog.dismiss();
            }
        });
        builder.setNegativeButton(context.getString(R.string.Nein),
                new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
                dialog.dismiss();
            }
        });
        builder.show();
    }

    @Override
    public void onLongClicked(boolean isActive, List<Person> personsToArchive) {
        mPersonsToArchiveOrDelete = personsToArchive;
        mIsLongClickActive = isActive;
        getActivity().invalidateOptionsMenu();
    }

    @Override
    public void onSoftDeleted(List<Person> softDeletedPersons) {
        mNumberOfSoftDeleted = softDeletedPersons.size();
        new FindAllActivePersonsFromDb(getContext(), this).execute();
    }

    @Override
    public void onDeletedPersons(Boolean isDeleted, List<Integer> deletedPositions) {
        // for sake of simplicity, this is skipped and set as nice feature, when the basics work
        iMainListFragment.onChosenPersonsDeleted(mPersonsToArchiveOrDelete.size(), isDeleted);
        /*
        //TODO: check this behaviour
        if (isDeleted) {
            for (Integer i : deletedPositions) {
                //TODO: the correct setting of the position is missing...
                // (what the callback in the else is starting)
                mMainListAdapter.notifyItemRemoved(i);
            }
        } else {
            //this only was existent before...
            iMainListFragment.onChosenPersonsDeleted(mPersonsToArchiveOrDelete.size(), isDeleted);
        }
        */
    }

    @Override
    public void onFindAllActive(List<Person> persons) {
        new SetPositionByNumberOfSoftDeletedPersonInDb(getContext(), this, persons).execute();
    }

    @Override
    public void onSetPosBySoftDelPers(List<Person> persons) {
        iMainListFragment.onChoosenPersonsSoftDeleted(mNumberOfSoftDeleted);
    }

    public interface IMainListFragment {
        void onChoosenPersonsSoftDeleted(int numberOfSoftDeletedPersons);
        void onChosenPersonsDeleted(int numberOfDeletedPersons, Boolean isDeleted);
        void onMenuEditButtonClicked(Person person);
    }
}
